package gov.va.med.mhv.vitals.web.controller;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;

import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.context.FacesContext;
import javax.faces.event.ComponentSystemEvent;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jfree.data.time.Day;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYDataset;
import org.primefaces.component.datatable.DataTable;
import org.primefaces.event.data.SortEvent;
import org.primefaces.model.StreamedContent;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.vitals.dto.BloodSugarReadingDTO;
import gov.va.med.mhv.vitals.enums.PeriodEnumeration;
import gov.va.med.mhv.vitals.util.CommonUtility;
import gov.va.med.mhv.vitals.web.util.WebUtility;

@ManagedBean
@Component
@Scope("session")
public class BloodSugarController extends AbstractController {

	private static final long serialVersionUID = -2722667818753652061L;
	private static Logger log = LogManager.getLogger(BloodSugarController.class);
	private List<BloodSugarReadingDTO> bsreadings = new ArrayList<BloodSugarReadingDTO>();
	private BloodSugarReadingDTO selectedBSReading = new BloodSugarReadingDTO();
	private BloodSugarReadingDTO newBSReading = new BloodSugarReadingDTO();
	private static final String TITLE = "Blood Sugar";
	private static final String XAXIS = "Blood Sugar Count";
	private static final String YAXIS = "mg/DL";

	public void init(ComponentSystemEvent event) {
		findUser();
		userprofileId = getUserProfileId();
		DataTable bsTable = (DataTable) FacesContext.getCurrentInstance().getViewRoot().findComponent("bloodSugarForm:bsugarList");
		if (!FacesContext.getCurrentInstance().isPostback()) {
			resetMessages();
			if (userprofileId != null) {
				bsreadings = findBSReadings(userprofileId);
			}
			setRowsPerPage(10);
			setPeriod(PeriodEnumeration.ONE_MONTH.getDescription());
		}
		else{
			if(sortColumn != null && sortBy != null){
				bsTable.setValueExpression("sortBy", sortColumn);
				bsTable.setSortOrder(sortBy);
			} 
		}
	}

	public void onSort(SortEvent event){
		sortColumn=event.getSortColumn().getValueExpression("sortBy");
		sortBy=event.isAscending()?"ascending":"descending";
	}
	
	private List<BloodSugarReadingDTO> findBSReadings(Long userprofileId) {
		List<BloodSugarReadingDTO> dtoList = null;
		try {

			dtoList = this.vitalSignsService.getBloodSugarReadingsForUser(userprofileId);

		} catch (Exception e) {
			log.error("Error in Find Blood Sugar readings:", e);
			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		} /*
			 * finally{ responseReader=null; }
			 */
		return dtoList;
	}

	public String showDetail(BloodSugarReadingDTO bloodSugarReadingDTO) {
		resetMessages();
		selectedBSReading = bloodSugarReadingDTO;
		setHourMinute(selectedBSReading);
		return "bloodSugar";
	}
	
	public String editDisplay(BloodSugarReadingDTO bloodSugarReadingDTO) {
		resetMessages();
		selectedBSReading = bloodSugarReadingDTO;
		setHourMinute(selectedBSReading);
		return "editBSDisplay";
	}
	
	public String resetEditDisplay(){
		return "editBSDisplay";
	}

	public String deleteDisplay(BloodSugarReadingDTO bloodSugarReadingDTO) {
		resetMessages();
		selectedBSReading = bloodSugarReadingDTO;
		setHourMinute(selectedBSReading);
		setDeleteOrigin("tableView");
		return "deleteBSDisplay";
	}

	public String deleteRecordDisplay() {
		resetMessages();
		setDeleteOrigin(null);
		return "deleteBSDisplay";
	}

	private void setHourMinute(BloodSugarReadingDTO selectedBSReading) {
		String dateTime = WebUtility.dateToString(selectedBSReading.getReading(), HOUR_MINUTE);
		if (!dateTime.equals(DEFAULT_DATETIME)) {
			selectedBSReading.setHour(WebUtility.dateToString(selectedBSReading.getReading(), HOURS));
			selectedBSReading.setMinute(WebUtility.dateToString(selectedBSReading.getReading(), MINUTES));
		}

	}

	public String addDisplay() {
		resetMessages();
		newBSReading = new BloodSugarReadingDTO();
		newBSReading.setReading(new Date());
		return "addBSDisplay";
	}

	public String dashboardAddDisplay() {
		resetMessages();
		findUser();
		setRowsPerPage(10);
		userprofileId = getUserProfileId();
		if (userprofileId != null) {
			bsreadings = findBSReadings(userprofileId);
		}
		newBSReading = new BloodSugarReadingDTO();
		newBSReading.setReading(new Date());
		FacesContext context = FacesContext.getCurrentInstance();
		context.getApplication().getNavigationHandler().handleNavigation(context, null,
				"/views/bloodsugar/addBSDisplay.xhtml");
		return null;
	}
	
	public String dashboardViewMore() {
		resetMessages();
		findUser();
		setRowsPerPage(10);
		userprofileId = getUserProfileId();
		if (userprofileId != null) {
			bsreadings = findBSReadings(userprofileId);
		}
		return "/views/bloodsugar/bloodSugarList";
	}

	public String showDahBoardDetail() {
		resetMessages();
		findUser();
		Long userProfileId = getUserProfileId();
		String bsid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("bsid");
		if (bsid != null && userProfileId != null) {
			Long id = Long.valueOf(bsid);
			selectedBSReading = getBSReadingById(userProfileId, id);
		}
		FacesContext context = FacesContext.getCurrentInstance();
		context.getApplication().getNavigationHandler().handleNavigation(context, null,
				"/views/bloodsugar/bloodSugar.xhtml");
		return null;
	}

	private BloodSugarReadingDTO getBSReadingById(Long userProfileId, Long id) {
		BloodSugarReadingDTO dto = null;
		try {

			dto = this.vitalSignsService.getBloodSugarReadingBy(getUserProfileId(), id);

		} catch (Exception e) {
			log.error("Error in Find BS reading By id:", e);
			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}
		return dto;
	}

	public String delete() {
		String outcome = null;
		String bsid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("bsid");
		Long userProfileId = getUserProfileId();

		try {
			if (bsid != null && userProfileId != null) {
				Long id = Long.valueOf(bsid);

				this.vitalSignsService.deleteBloodSugarReading(userProfileId, id);
				bsreadings = findBSReadings(userprofileId);
				deleteMessage = true;
				outcome = "bloodSugarList";
			}
		} catch (Exception e) {
			log.error("Failed to Delete Blood Sugar record", e);

			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}

		return outcome;
	}

	public String save() {
		String outcome = null;
		try {
			if (saveBSReading().getBloodSugarId() != null) {
				outcome = "bloodSugarList";
				bsreadings = findBSReadings(userprofileId);
				saveMessage = true;
			}
		} catch (MHVException e) {
			log.error("Failed to Save Blood Sugar record", e);
			if (e.getValidationErrorMessages() != null) {
				addValidationMessages(e.getValidationErrorMessages());
			} else {
				FacesContext.getCurrentInstance().addMessage(null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
		return outcome;
	}

	public BloodSugarReadingDTO saveBSReading() throws MHVException {
		resetMessages();
		String bsid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("bsid");
		Long id = null;
		BloodSugarReadingDTO bsreading = null;
		if (bsid != null) {
			id = Long.valueOf(bsid);
			bsreading = getSelectedBSReading();
		} else {
			bsreading = getNewBSReading();
		}

		prepareBSReading(id, bsreading);
		Long bsrId = this.vitalSignsService.saveBloodSugarReading(bsreading);
		bsreading.setBloodSugarId(bsrId);

		return bsreading;
	}

	private BloodSugarReadingDTO prepareBSReading(Long id, BloodSugarReadingDTO bsreading) {
		Date dateTime = null;
		if (bsreading.getReading() != null) {
			if (bsreading.getHour() == null && bsreading.getMinute() == null) {
				dateTime = WebUtility.getDaTeTime(bsreading.getReading(), DEFAULT_HOUR, DEFAULT_MINUTE);
			} else {
				dateTime = WebUtility.getDaTeTime(bsreading.getReading(), bsreading.getHour(), bsreading.getMinute());
			}
		}
		bsreading.setReading(dateTime);
		bsreading.setUserprofileId(getUserProfileId());
		return bsreading;
	}

	public String saveAndAdd() {
		String outcome = null;
		try {
			if (saveBSReading().getBloodSugarId() != null) {
				outcome = "addBSDisplay";
				saveAndAddMessage = true;
				bsreadings = findBSReadings(userprofileId);
				newBSReading = new BloodSugarReadingDTO();
				newBSReading.setReading(new Date());
			}

		} catch (MHVException e) {
			log.error("Failed to Save Blood Sugar record", e);
			if (e.getValidationErrorMessages() != null) {
				addValidationMessages(e.getValidationErrorMessages());
			} else {
				FacesContext.getCurrentInstance().addMessage(null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
		return outcome;
	}

	public String updateLineModel() {
		return "bloodSugarGraph";
	}

	public StreamedContent getJchart() {
		StreamedContent content = null;
		try {
			content = createChart(TITLE, XAXIS, YAXIS);
		} catch (Exception e) {
			log.error("Failed to create Chart", e);
		}
		return content;
	}

	protected XYDataset createDataset(Set<String> uniqueYears) {
		TimeSeriesCollection dataset = new TimeSeriesCollection();
		TimeSeries series1 = new TimeSeries(YAXIS);
		String period = getPeriod();
		Date startDate = null;
		List<BloodSugarReadingDTO> graphList;
		if (!period.equals(PeriodEnumeration.ALL_DATA.getDescription())) {
			startDate = getStartGraphDate(period);
		}
		if (startDate != null) {
			graphList = getGraphlist(bsreadings, startDate);
		} else {
			graphList = bsreadings;
		}
		for (BloodSugarReadingDTO bsreading : graphList) {

			series1.addOrUpdate(new Day(bsreading.getReading()), bsreading.getBloodSugarCount());
			uniqueYears
					.add(CommonUtility.dateToString(bsreading.getReading(), YYYY_MM_DD_HHMMSS_FORMAT).substring(0, 4));
		}
		dataset.addSeries(series1);
		return dataset;
	}

	private List<BloodSugarReadingDTO> getGraphlist(List<BloodSugarReadingDTO> bsreadings, Date startDate) {
		List<BloodSugarReadingDTO> graphList = new ArrayList<BloodSugarReadingDTO>();
		Integer startdateValue = Integer.valueOf(CommonUtility.dateToString(startDate, YYYYMMDD_FORMAT));
		for (BloodSugarReadingDTO bsreading : bsreadings) {
			Integer readingValue = Integer.valueOf(CommonUtility.dateToString(bsreading.getReading(), YYYYMMDD_FORMAT));
			if (readingValue >= startdateValue) {
				graphList.add(bsreading);
			}
		}
		return graphList;
	}

	public String printerFriendlySummary(){
		resetMessages();
		return "printBSSummary";
	}

	public List<BloodSugarReadingDTO> getBsreadings() {
		return bsreadings;
	}

	public void setBsreadings(List<BloodSugarReadingDTO> bsreadings) {
		this.bsreadings = bsreadings;
	}

	public BloodSugarReadingDTO getSelectedBSReading() {
		return selectedBSReading;
	}

	public void setSelectedBSReading(BloodSugarReadingDTO selectedBSReading) {
		this.selectedBSReading = selectedBSReading;
	}

	public BloodSugarReadingDTO getNewBSReading() {
		return newBSReading;
	}

	public void setNewBSReading(BloodSugarReadingDTO newBSReading) {
		this.newBSReading = newBSReading;
	}

}
